#include "SerialCommHelper.h"
#include <Process.h>
#include <assert.h>
#include <atlbase.h>
#include <iostream>

/// <summary>
/// This function is used to invalidates the handle when no longer needed.
/// </summary>
/// <param name="hHandle">Handle to invalidate</param>
void CSerialCommHelper::InvalidateHandle(HANDLE& hHandle)
{
	hHandle = INVALID_HANDLE_VALUE;
}

/// <summary>
/// This function tries to close the handle and then invalidates it.
/// </summary>
/// <param name="hHandle">Handle to be closed</param>
void CSerialCommHelper::CloseAndCleanHandle(HANDLE& hHandle)
{
	BOOL abRet = CloseHandle(hHandle);
	if (!abRet)
	{
		assert(0);
	}
	InvalidateHandle(hHandle);
}

/// <summary>
/// This function receives raw position data from read thread and parses each 
/// coordinate i.e. S,X,Y and Z into position structure and forwards it to the
/// subscriber.
/// </summary>
/// <param name="data">Raw mouse data string</param>
void CSerialCommHelper::SetPosition(std::string data)
{
	try {
		const char* arr = data.c_str();
		const char* ms;
		int temp;
		StPosition newPosition;

		// Find and parse S component of data.
		//ms = strstr(arr, "S");
		//if (ms != NULL) {
		//	if (sscanf_s(ms, "S%d", &temp) == 1) {
		//		newPosition.S = temp;
		//	}
		//}

		//// Find and parse X component of data.
		//ms = strstr(arr, "X");
		//if (ms != NULL) {
		//	if (sscanf_s(ms, "X%d", &temp) == 1) {
		//		newPosition.X = temp;
		//	}
		//}

		//// Find and parse Y component of data.
		//ms = strstr(arr, "Y");
		//if (ms != NULL) {
		//	if (sscanf_s(ms, "Y%d", &temp) == 1) {
		//		newPosition.Y = temp;
		//	}
		//}

		//// Find and parse Z component of data.
		//ms = strstr(arr, "Z");
		//if (ms != NULL) {
		//	if (sscanf_s(ms, "Z%d", &temp) == 1) {
		//		newPosition.Z = temp;
		//	}
		//}
		int x, y;
		int newx, newy, newz;

		x = get_int(arr[7], arr[8]);
		y = get_int(arr[9], arr[10]);
		newx = x + y; // the data is sent rotated to emulate the Immersion Mouse, so unrotate it for diaplay
		newy = y - x;
		newz = get_int(arr[11], arr[12]); // Z is normal
		newPosition.X = newx;
		newPosition.Y = newy;
		newPosition.Z = newz;



		// Forward new position data to subscriber.
		if (dCallback != 0)
			dCallback(newPosition);
	}
	catch (...) {
		// Do nothing.
	}
}

int CSerialCommHelper::get_int(int a, int b) { // convert to int
	int temp = (a << 7) + b; // a are the upper 7 bits, b the lower
	if (temp > 8192) temp -= 16384; // large numbers are negitive
	return temp;
}

/// <summary>
/// CSerialCommHelper Constructor.
/// </summary>
CSerialCommHelper::CSerialCommHelper()
{
	m_streamingDataStarted = false;
	InvalidateHandle(m_hThreadTerm);
	InvalidateHandle(m_hThread);
	InvalidateHandle(m_hThreadStarted);
	InvalidateHandle(m_hCommPort);
	InvalidateHandle(m_hDataRx);

	InitLock();
	m_eState = SS_UnInit;
}

/// <summary>
/// CSerialCommHelper Desconstructor.
/// </summary>
CSerialCommHelper::~CSerialCommHelper()
{
	m_eState = SS_Unknown;
	DelLock();
}

/// <summary>
/// Sets up initial parameters for the device such data/stop bits, partiy, flow control and 
/// timeouts.
/// </summary>
/// <returns>TRUE if successful, otherwise returns FALSE.</returns>
HRESULT CSerialCommHelper::Init(const char* szPortName, DWORD dwBaudRate, BYTE byParity, BYTE byStopBits, BYTE byByteSize)
{
	HRESULT hr = S_OK;
	try
	{
		size_t size = strlen(szPortName) + 1;
		wchar_t* portName = new wchar_t[size];

		size_t outSize;
		mbstowcs_s(&outSize, portName, size, szPortName, size - 1);

		m_hDataRx = CreateEvent(0, 0, 0, 0);

		m_hCommPort = ::CreateFile(portName,
			GENERIC_READ | GENERIC_WRITE,//access ( read and write)
			0,	//(share) 0:cannot share the COM port						
			0,	//security  (None)				
			OPEN_EXISTING,// creation : open_existing
			FILE_FLAG_OVERLAPPED,// we want overlapped operation
			0// no templates file for COM port...
		);
		if (m_hCommPort == INVALID_HANDLE_VALUE)
		{
			return E_FAIL;
		}

		if (!::SetCommMask(m_hCommPort, EV_RXCHAR | EV_TXEMPTY))
		{
			return E_FAIL;
		}

		DCB dcb = { 0 };
		dcb.DCBlength = sizeof(DCB);

		if (!::GetCommState(m_hCommPort, &dcb))
		{
			return E_FAIL;
		}

		dcb.BaudRate = dwBaudRate;
		dcb.ByteSize = byByteSize;
		dcb.Parity = byParity;
		if (byStopBits == 1)
			dcb.StopBits = ONESTOPBIT;
		else if (byStopBits == 2)
			dcb.StopBits = TWOSTOPBITS;
		else
			dcb.StopBits = ONE5STOPBITS;

		dcb.fBinary = TRUE;
		dcb.fDtrControl = RTS_CONTROL_DISABLE;
		dcb.fRtsControl = RTS_CONTROL_DISABLE;
		dcb.fOutxCtsFlow = FALSE;
		dcb.fOutxDsrFlow = FALSE;
		dcb.fDsrSensitivity = FALSE;
		dcb.fAbortOnError = TRUE;


		if (!::SetCommState(m_hCommPort, &dcb))
		{
			return E_FAIL;
		}

		COMMTIMEOUTS timeouts;
		timeouts.ReadIntervalTimeout = MAXDWORD;
		timeouts.ReadTotalTimeoutMultiplier = 0;
		timeouts.ReadTotalTimeoutConstant = 0;
		timeouts.WriteTotalTimeoutMultiplier = 0;
		timeouts.WriteTotalTimeoutConstant = 0;

		if (!SetCommTimeouts(m_hCommPort, &timeouts))
		{
			return E_FAIL;
		}





		m_hThreadTerm = CreateEvent(0, 0, 0, 0);
		m_hThreadStarted = CreateEvent(0, 0, 0, 0);

		m_hThread = (HANDLE)_beginthreadex(0, 0, CSerialCommHelper::ThreadFn, (void*)this, 0, 0);

		DWORD dwWait = WaitForSingleObject(m_hThreadStarted, INFINITE);

		assert(dwWait == WAIT_OBJECT_0);

		CloseHandle(m_hThreadStarted);

		InvalidateHandle(m_hThreadStarted);
		m_abIsConnected = true;

		
	}
	catch (...)
	{
		assert(0);
		hr = E_FAIL;
	}
	if (SUCCEEDED(hr))
	{
		m_eState = SS_Init;
	}
	return hr;
}

/// <summary>
/// This function sets state as Started.
/// </summary>
/// <returns></returns>
HRESULT CSerialCommHelper::Start()
{
	m_eState = SS_Started;

	init_IMMC();

	Write("BEGIN", 5);
	std::string inbuff;
	int num = Read_N(inbuff, 4, 50);
	Sleep(1000);
	std::cout << "Send streaming command now. " << std::endl;
	char st[25] = { 0xCF, 0x00, 0x64, 0x0A, 0x0F, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01 };

	Write(st, 25);
	num = Read_N(inbuff, 1, 50);

	m_streamingDataStarted = true;
	return S_OK;
}

/// <summary>
/// This function sets state as Stopped.
/// </summary>
/// <returns></returns>
HRESULT CSerialCommHelper::Stop()
{
	m_eState = SS_Stopped;
	return S_OK;
}

/// <summary>
/// This function closes the read thread and removes the subscriber callback.
/// Its automatically called when Close() is called.
/// </summary>
/// <returns>Returns S_OK if successful, E_FAIL if fails.</returns>
HRESULT CSerialCommHelper::UnInit()
{
	HRESULT hr = S_OK;
	try
	{
		m_streamingDataStarted = false;
		Write("E", 1);
		m_abIsConnected = false;
		SignalObjectAndWait(m_hThreadTerm, m_hThread, INFINITE, FALSE);
		CloseAndCleanHandle(m_hThreadTerm);
		CloseAndCleanHandle(m_hThread);
		CloseAndCleanHandle(m_hDataRx);
		CloseAndCleanHandle(m_hCommPort);
	}
	catch (...)
	{
		assert(0);
		hr = E_FAIL;
	}
	if (SUCCEEDED(hr))
		m_eState = SS_UnInit;

	dCallback = nullptr;

	return hr;
}

/// <summary>
/// Its a thread function that runs in a loop and waits for a COMM event. If finds any new bytes to reads,
/// and forwards the string to SetPosition function.
/// </summary>
/// <param name="pvParam">CSerialCommHelper object to receive data updates through SetPosition function.</param>
/// <returns></returns>
unsigned __stdcall CSerialCommHelper::ThreadFn(void* pvParam)
{
	CSerialCommHelper* apThis = (CSerialCommHelper*)pvParam;
	bool abContinue = true;
	DWORD dwEventMask = 0;

	OVERLAPPED ov;
	memset(&ov, 0, sizeof(ov));
	ov.hEvent = CreateEvent(0, true, 0, 0);
	HANDLE arHandles[2];
	arHandles[0] = apThis->m_hThreadTerm;

	DWORD dwWait;
	SetEvent(apThis->m_hThreadStarted);
	while (abContinue)
	{
		BOOL abRet = ::WaitCommEvent(apThis->m_hCommPort, &dwEventMask, &ov);
		if (!abRet)
		{
			assert(GetLastError() == ERROR_IO_PENDING);
		}

		arHandles[1] = ov.hEvent;

		dwWait = WaitForMultipleObjects(2, arHandles, FALSE, INFINITE);
		switch (dwWait)
		{
		case WAIT_OBJECT_0:
		{
			_endthreadex(1);
		}
		break;
		case WAIT_OBJECT_0 + 1:
		{
			DWORD dwMask;
			if (GetCommMask(apThis->m_hCommPort, &dwMask))
			{
				if (dwMask == EV_TXEMPTY)
				{
					ResetEvent(ov.hEvent);
					continue;
				}
			}

			int iAccum = 0;
			apThis->m_theSerialBuffer.LockBuffer();

			try
			{
				std::string szDebug;
				BOOL abRet = false;

				DWORD dwBytesRead = 0;
				bool notEndChar = true;
				OVERLAPPED ovRead;
				memset(&ovRead, 0, sizeof(ovRead));
				ovRead.hEvent = CreateEvent(0, true, 0, 0);

				do
				{
					ResetEvent(ovRead.hEvent);
					char szTmp[1];
					int iSize = sizeof(szTmp);
					memset(szTmp, 0, sizeof szTmp);
					abRet = ::ReadFile(apThis->m_hCommPort, szTmp, sizeof(szTmp), &dwBytesRead, &ovRead);
					if (!abRet)
					{
						abContinue = FALSE;
						break;
					}
					if (dwBytesRead > 0)
					{
						//std::cout<< szTmp <<std::endl;
						apThis->m_theSerialBuffer.AddData(szTmp, dwBytesRead);
						szDebug.append(szTmp, dwBytesRead);
						iAccum += dwBytesRead;
					}

				} while (dwBytesRead > 0 || (apThis->m_streamingDataStarted && szDebug.length() < 13));

				//std::cout << szDebug << std::endl;

				if (szDebug.length() > 0) {
					//apThis->print_hex(szDebug);
					if (szDebug.length() == 13) {
						apThis->SetPosition(szDebug);
					}
				}

				CloseHandle(ovRead.hEvent);
			}
			catch (...)
			{
				assert(0);
			}

			//if we are not in started state then we should flush the queue...( we would still read the data)
			if (apThis->GetCurrentState() != SS_Started)
			{
				iAccum = 0;
				apThis->m_theSerialBuffer.Flush();
			}

			apThis->m_theSerialBuffer.UnLockBuffer();

			if (iAccum > 0)
			{
				apThis->SetDataReadEvent();
			}
			ResetEvent(ov.hEvent);
		}
		break;
		}
	}
	return 0;
}

void CSerialCommHelper::print_hex(std::string string)
{
	//unsigned char* p = (unsigned char*)string;

	for (int i = 0; i < string.length(); ++i) {
		if (!(i % 16) && i)
			printf("\n");

		printf("0x%02x ", string[i]);
	}
	printf("\n\n");
}


/// <summary>
/// This functions check current state and returns S_OK if communication can happen, otherwise 
/// </summary>
/// <returns></returns>
HRESULT  CSerialCommHelper::CanProcess()
{
	switch (m_eState)
	{
	case SS_Unknown:assert(0); return E_FAIL;
	case SS_UnInit:return E_FAIL;
	case SS_Started:return S_OK;
	case SS_Init:
	case SS_Stopped:
		return E_FAIL;
	default:assert(0);
	}
	return E_FAIL;
}

/// <summary>
/// This function is used to subscribe to the data updates.
/// </summary>
/// <param name="cB">Callback function of void return type and one input parameter of type StPosition</param>
void CSerialCommHelper::setDCallback(type_myCallBack cB)
{
	dCallback = cB;
}

/// <summary>
/// This function is used to write data to COM port.
/// </summary>
/// <param name="data">Data to write</param>
/// <param name="dwSize">Size of data to write</param>
/// <returns></returns>
HRESULT CSerialCommHelper::Write(const char* data, DWORD dwSize)
{
	HRESULT hr = CanProcess();
	if (FAILED(hr)) return hr;
	int iRet = 0;
	OVERLAPPED ov;
	memset(&ov, 0, sizeof(ov));
	ov.hEvent = CreateEvent(0, true, 0, 0);
	DWORD dwBytesWritten = 0;
	//do
	{
		iRet = WriteFile(m_hCommPort, data, dwSize, &dwBytesWritten, &ov);
		if (iRet == 0)
		{
			WaitForSingleObject(ov.hEvent, INFINITE);
		}

	}//	while ( ov.InternalHigh != dwSize ) ;
	CloseHandle(ov.hEvent);
	std::string szData(data);

	std::cout << ">> Wrote : " << data << std::endl;
	return S_OK;
}


HRESULT CSerialCommHelper::Read_N(std::string& data, long alCount, long  alTimeOut)
{
	HRESULT hr = CanProcess();

	if (FAILED(hr))
	{
		//ASSERT(0);
		return hr;
	}

	////ATLTRACE6("CSerialCommHelper : CSerialCommHelper : Read_N called for %d bytes",alCount);
	try
	{

		std::string szTmp;
		szTmp.erase();

		//ATLTRACE6("CSerialCommHelper : CSerialCommHelper : Read_N (%d) locking the queue  ",alCount);

		int iLocal = m_theSerialBuffer.Read_N(szTmp, alCount, m_hDataRx);

		if (iLocal == alCount)
		{
			data = szTmp;
		}
		else
		{//there are either none or less bytes...
			long iRead = 0;
			int iRemaining = alCount - iLocal;
			while (1)
			{

				//ATLTRACE6("CSerialCommHelper : CSerialCommHelper : Read_N (%d) making blocking read() ",alCount);

				DWORD dwWait = WaitForSingleObject(m_hDataRx, alTimeOut);

				if (dwWait == WAIT_TIMEOUT)
				{
					std::cout << "Timeout." << std::endl;
					//ATLTRACE6("CSerialCommHelper : CSerialCommHelper : Read_N (%d) timed out in blocking read",alCount);
					data.erase();
					hr = E_FAIL;
					return hr;

				}

				//ASSERT ( dwWait == WAIT_OBJECT_0 );
				//ATLTRACE6("CSerialCommHelper : CSerialCommHelper : Read_N (%d) Woke Up from WaitXXX() locking Q",alCount);


				iRead = m_theSerialBuffer.Read_N(szTmp, iRemaining, m_hDataRx);
				iRemaining -= iRead;

				//ATLTRACE6("CSerialCommHelper : CSerialCommHelper : Read_N (%d) Woke Up from WaitXXX() Unlocking Q",alCount);


				if (iRemaining == 0)
				{
					//ATLTRACE6("CSerialCommHelper : CSerialCommHelper : Read_N (%d) Woke Up from WaitXXX() Done reading ",alCount);
					data = szTmp;
					std::cout << "<< Read : " << data << std::endl;
					return S_OK;
				}
			}
		}
	}
	catch (...)
	{
		//ATLTRACE6(_T("CSerialCommHelper Unhandled exception"));
		//ASSERT ( 0  ) ;
		std::cout << "Exception." << std::endl;
	}
	
	return hr;

}

/// <summary>
/// Reads registry keys for FTDIBUS and enumerates COM ports.
/// </summary>
/// <returns>Returns true if successful, returns false if fails.</returns>
bool CSerialCommHelper::detectPort()
{
	ports.clear();

	ATL::CRegKey FTDIBUSKey;
	LSTATUS nStatus{ FTDIBUSKey.Open(HKEY_LOCAL_MACHINE, _T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS"), KEY_ENUMERATE_SUB_KEYS) };
	if (nStatus != ERROR_SUCCESS)
	{
		SetLastError(nStatus);
		return false;
	}

	TCHAR deviceSubkey[512];
	DWORD dskLength = 512;
	int i = 0;

	while (true)
	{
		nStatus = FTDIBUSKey.EnumKey(i, deviceSubkey, &dskLength);
		if (nStatus == ERROR_SUCCESS)
		{
			ATL::CAtlString RootAndDevice;
			RootAndDevice.Format(_T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS\\%s"), deviceSubkey);

			ATL::CRegKey DeviceKey;
			nStatus = DeviceKey.Open(HKEY_LOCAL_MACHINE, RootAndDevice, KEY_ENUMERATE_SUB_KEYS);
			if (nStatus != ERROR_SUCCESS)
			{
				SetLastError(nStatus);
				return false;
			}

			TCHAR DIndexSubkey[512];
			DWORD diskLength = 512;
			int j = 0;

			while (true)
			{
				nStatus = DeviceKey.EnumKey(j, DIndexSubkey, &diskLength);
				if (nStatus == ERROR_SUCCESS)
				{
					ATL::CAtlString RootAndDeviceAndIndex;
					RootAndDeviceAndIndex.Format(_T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS\\%s\\%s"), deviceSubkey, DIndexSubkey);

					ATL::CRegKey DeviceIndexKey;
					nStatus = DeviceIndexKey.Open(HKEY_LOCAL_MACHINE, RootAndDeviceAndIndex, KEY_ENUMERATE_SUB_KEYS);
					if (nStatus != ERROR_SUCCESS)
					{
						SetLastError(nStatus);
						return false;
					}

					TCHAR DeviceParamsSubkey[512];
					DWORD dpLength = 512;
					int k = 0;
					while (true)
					{
						nStatus = DeviceIndexKey.EnumKey(k, DeviceParamsSubkey, &dpLength);
						if (nStatus == ERROR_SUCCESS)
						{
							ATL::CAtlString DeviceParams;
							DeviceParams.Format(_T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS\\%s\\%s\\%s"), deviceSubkey, DIndexSubkey, DeviceParamsSubkey);

							ATL::CRegKey DeviceParamsKey;
							nStatus = DeviceParamsKey.Open(HKEY_LOCAL_MACHINE, DeviceParams, KEY_QUERY_VALUE);
							if (nStatus != ERROR_SUCCESS)
							{
								SetLastError(nStatus);
								return false;
							}

							TCHAR szPath[MAX_PATH] = _T("");
							ULONG nPathChars = _countof(szPath);
							nStatus = DeviceParamsKey.QueryStringValue(_T("PortName"), szPath, &nPathChars);
							if (nStatus == ERROR_SUCCESS)
							{
								std::wstring ws(szPath);
								// As the port name will not have unicode characters. 
#pragma warning(suppress: 4244)
								std::string str(ws.begin(), ws.end());
								ports.push_back(str);
								return true;
							}
						}
						k++;
					}
				}
				j++;
			}
		}
		else
			break;
		i++;
	}

	return false;
}

int CSerialCommHelper::init_IMMC()
{
	std::cout << "IMMC" << std::endl;
	char outbuff[] = "IMMC";
	char cr[] = "\r";
	std::string inbuff;
	int i, num;

	for (i = 0; i < 20; i++)
	{
		Write(cr, strlen(cr)); // if sending cr returns cr, then the mouse is Z-Type
		num = Read_N(inbuff, 4, 50); // so return -99
		if (inbuff.find("\r") != std::string::npos) return -99; // Z-Type returns Vxx<cr>, so we can test for V or cr
	}

	for (i = 0; i < 20; i++)
	{
		Write(outbuff, 4);
		num = Read_N(inbuff, 4, 50);
		if (inbuff.find("C") != std::string::npos) return 1;
	}
	return 0;
}

